BriefingScreen = {}

BriefingUIState = {
    isActive = false,
    isCompleted = false
}

local function delayedCallback(delay, func)
    local endTime = os.time() + delay
    local function check()
        if os.time() >= endTime then
            func(); Events.OnTick.Remove(check)
        end
    end
    Events.OnTick.Add(check)
end

local VolumeTracker = {
    isActive = false,
    duration = 3,
    originalVolume = 0,
    volumeRestored = false,
}

function VolumeTracker.onKeyPress(key)
    if key == Keyboard.KEY_ESCAPE then
        VolumeTracker:restoreVolume()
    end
    return false
end

function VolumeTracker:start(originalVolume)
    self.isActive = true
    self.originalVolume = originalVolume
    self.volumeRestored = false

    delayedCallback(self.duration, function()
        VolumeTracker:restoreVolume()
    end)
    
    Events.OnKeyPressed.Add(VolumeTracker.onKeyPress)
end

function VolumeTracker:restoreVolume()
    if self.isActive and not self.volumeRestored then
        getCore():setOptionMusicVolume(self.originalVolume)
        Events.OnKeyPressed.Remove(VolumeTracker.onKeyPress)
        self.volumeRestored = true
        self.isActive = false
    end
end

local OnScreenIntroUI = ISPanel:derive("OnScreenIntroUI")

local musicVolume = getCore():getOptionMusicVolume()

function OnScreenIntroUI:new(lines)
    local o = ISPanel.new(self, 0, 0, getCore():getScreenWidth(), getCore():getScreenHeight())
    o.lines = lines or {}
    o:noBackground()
    o.borderColor = {r=0, g=0, b=0, a=0}
    o:setAlwaysOnTop(true)

    o:setWantKeyEvents(true)
    o:setCapture(true)

    o.holdDuration = 1.0
    o.fadeOutDuration = 1.0
    o.typingDelay = 1.0

    o.currentLineIndex = 1
    o.currentCharIndex = 0
    o.isTypingComplete = false
    o.holdStartTime = nil
    o.fadeStartTime = nil
    o.startTime = nil
    o.typingStarted = false
    o.alpha = 1.0

    o.showCursor = true

    o.musicVolumeRestored = false
    o.musicVolumeLowered = false

    o.skipRequested = false

    o.font = UIFont.Large
    o.fontHgt = getTextManager():getFontHeight(o.font)

    local maxWidth = 0
    for _, line in ipairs(o.lines) do
        maxWidth = math.max(maxWidth, getTextManager():MeasureStringX(o.font, line))
    end
    o.textBlockWidth = maxWidth + 40
    o.textBlockHeight = #o.lines * (o.fontHgt + 5) + 20

    o.startTime = getTimestampMs()

    o.isMiseryTimeSkipActive = false
    
    BriefingUIState.isActive = true
    BriefingUIState.isCompleted = false

    return o
end

function OnScreenIntroUI:skipAnimation()
    if self.fadeStartTime then
        return
    end
    
    self.skipRequested = true
    
    if not self.isTypingComplete then
        self.currentLineIndex = #self.lines + 1
        self.currentCharIndex = 0
        self.isTypingComplete = true
        self.holdStartTime = getTimestampMs()
        self.showCursor = false
    end
    
    self.fadeStartTime = getTimestampMs()
end

function OnScreenIntroUI:onKeyPress(key)
    if (key == Keyboard.KEY_ESCAPE) and not self.isMiseryTimeSkipActive then
        GameKeyboard.eatKeyPress(key); self:skipAnimation()
        return true
    elseif key == Keyboard.KEY_ESCAPE and self.isMiseryTimeSkipActive then
        GameKeyboard.eatKeyPress(key);
        return true
    end
    return false
end

function OnScreenIntroUI:onMouseDown()
    if not self.isMiseryTimeSkipActive then
        self:skipAnimation()
        return true
    end
    return false
end

function OnScreenIntroUI:onMouseUp()
    return true
end

function OnScreenIntroUI:onRightMouseDown(x, y)
    if not self.isMiseryTimeSkipActive then
        self:skipAnimation()
        return true
    end
    return false
end

function OnScreenIntroUI:onRightMouseUp(x, y)
    return true
end

function OnScreenIntroUI:updateCursor()
    if self.isTypingComplete then
        self.showCursor = false
        return
    end
    
    self.showCursor = true
end

function OnScreenIntroUI:updateTyping()
    if self.isTypingComplete or self.skipRequested then return end

    if not self.typingStarted then
        local adjustedTime = getTimestampMs()
        local timeSinceStart = (adjustedTime - self.startTime) / 1000.0
        
        if timeSinceStart >= self.typingDelay then
            self.typingStarted = true
        else
            return
        end
    end

    self.currentCharIndex = self.currentCharIndex + 1
    
    if self.currentLineIndex <= #self.lines then
        local currentLine = self.lines[self.currentLineIndex]
        if self.currentCharIndex > #currentLine then
            self.currentLineIndex = self.currentLineIndex + 1
            self.currentCharIndex = 0
        end
    else
        self.isTypingComplete = true
        self.holdStartTime = getTimestampMs()
    end
end

function OnScreenIntroUI:updateFadeOut()
    if not self.fadeStartTime then return end
    
    local adjustedTime = getTimestampMs()
    local fadeElapsed = (adjustedTime - self.fadeStartTime) / 1000.0
    local fadeProgress = fadeElapsed / self.fadeOutDuration
    
    self.alpha = math.max(0.0, 1.0 - fadeProgress)
    
    if fadeElapsed >= self.fadeOutDuration then
        self:removeFromUIManager()
        
        BriefingUIState.isCompleted = true
        BriefingUIState.isActive = false
        
        return true
    end
    
    return false
end

function OnScreenIntroUI:update()
    ISPanel.update(self)
    
    self:updateCursor()

    if getCore():getGameMode() == "Misery" and MiseryTimeSkip then
        self.isMiseryTimeSkipActive = MiseryTimeSkip.State.isActive
    end

    if not self.isMiseryTimeSkipActive then
        if getGameSpeed() > 0 and not self.fadeStartTime then
            setGameSpeed(0); UIManager.setShowPausedMessage(false)
        end
    end

    if not self.isTypingComplete and not self.skipRequested then
        self:updateTyping()
        return
    end

    if self.holdStartTime and not self.fadeStartTime and not self.skipRequested then
        local adjustedTime = getTimestampMs()
        local holdElapsed = (adjustedTime - self.holdStartTime) / 1000.0
        
        if holdElapsed >= self.holdDuration then
            self.fadeStartTime = adjustedTime
        end
    end
    
    if self.fadeStartTime and getGameSpeed() == 0 then
        if not self.isMiseryTimeSkipActive or (MiseryTimeSkip and MiseryTimeSkip.State.isCompleted) then
            if not getActivatedMods():contains("\\PauseStart") and not getActivatedMods():contains("\\Pause_on_Start") then
                setGameSpeed(1); UIManager.setShowPausedMessage(true)
                if not getCore():getOptionPlayMusicWhenPaused() then
                    getCore():setOptionMusicVolume(0); self.musicVolumeLowered = true
                    VolumeTracker:start(musicVolume)
                end
            else
                UIManager.setShowPausedMessage(true)
            end
        end
    end

    if self.fadeStartTime then
        self:updateFadeOut()
    end
end

function OnScreenIntroUI:getDisplayText(lineIndex)
    if not self.typingStarted and not self.skipRequested then
        return ""
    end
    
    local baseText = ""
    
    if not self.isTypingComplete and not self.skipRequested then
        if lineIndex < self.currentLineIndex then
            baseText = self.lines[lineIndex]
        elseif lineIndex == self.currentLineIndex then
            baseText = string.sub(self.lines[lineIndex], 1, self.currentCharIndex)
            if self.showCursor and lineIndex == self.currentLineIndex and self.currentCharIndex < #self.lines[lineIndex] then
                baseText = baseText .. "_"
            end
        else
            return ""
        end
    else
        baseText = self.lines[lineIndex]
    end
    
    return baseText
end

function OnScreenIntroUI:render()
    ISPanel.render(self)

    local actualAlpha = math.max(0.0, math.min(1.0, self.alpha))
    
    self:drawRect(0, 0, self.width, self.height, actualAlpha, 0, 0, 0)

    local textY = self.height / 2 - self.textBlockHeight / 2

    for i, _ in ipairs(self.lines) do
        local lineToDraw = self:getDisplayText(i)
        
        if #lineToDraw > 0 then
            local centerX = self.width / 2
            self:drawTextCentre(lineToDraw, centerX + 2, textY + 12, 0, 0, 0, actualAlpha * 0.7, self.font)
            self:drawTextCentre(lineToDraw, centerX, textY + 10, 1, 1, 1, actualAlpha, self.font)
        end
        
        textY = textY + self.fontHgt + 5
    end
end

function BriefingScreen.GetLocationName(x, y)
    if x >= 5500 and x <= 7399 and y >= 4535 and y <= 6169 then
        return getText("UI_Map_Riverside") .. ", " .. getText("UI_Map_KnoxCounty")
    elseif x >= 10320 and x <= 12889 and y >= 6000 and y <= 7969 then
        return getText("UI_Map_Westpoint") .. ", " .. getText("UI_Map_KnoxCounty")
    elseif x >= 7400 and x <= 9104 and y >= 10640 and y <= 12639 then
        return getText("UI_Map_Rosewood") .. ", " .. getText("UI_Map_KnoxCounty")
    elseif x >= 10040 and x <= 11717 and y >= 8740 and y <= 11196 then
        return getText("UI_Map_Muldraugh") .. ", " .. getText("UI_Map_KnoxCounty")
    elseif x >= 9200 and x <= 11079 and y >= 11970 and y <= 13699 then
        return getText("UI_Map_MarchRidge") .. ", " .. getText("UI_Map_KnoxCounty")
    elseif x >= 11700 and x <= 14699 and y >= 900 and y <= 3599 then
        return getText("UI_Map_Louisville") .. ", " .. getText("UI_Map_KnoxCounty")
    end
    
    return getText("UI_Map_KnoxCounty") .. ", " .. getText("UI_Map_Kentucky")
end

function BriefingScreen.FormatTime(hourOfDay)
    local hour = math.floor(hourOfDay)
    local minute = math.floor((hourOfDay - hour) * 60)
    
    if getCore():getOptionClock24Hour() then
        return string.format("%02d:%02d", hour, minute)
    else
        local ampm = getText("UI_Time_AM")
        
        if hour >= 12 then
            ampm = getText("UI_Time_PM")
            if hour > 12 then
                hour = hour - 12
            end
        end
        if hour == 0 then
            hour = 12
        end
        
        return string.format("%02d:%02d %s", hour, minute, ampm)
    end
end

function BriefingScreen.GetDaysSinceOutbreak(currentDay, currentMonth, currentYear, timeSinceApo)
    local function isLeapYear(year)
        return (year % 4 == 0 and year % 100 ~= 0) or (year % 400 == 0)
    end
 
    local function getDaysInMonth(month, year)
        local daysInMonth = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}
        if month == 2 and isLeapYear(year) then
            return 29
        end
        return daysInMonth[month]
    end
 
    local function dayOfYear(year, month, day)
        local days = day
        for m = 1, month - 1 do
            days = days + getDaysInMonth(m, year)
        end
        return days
    end
 
    local startYear = 1993
    local startMonth = 7
    local startDay = 4
 
    if currentYear < startYear or (currentYear == startYear and currentMonth < startMonth) or (currentYear == startYear and currentMonth == startMonth and currentDay < startDay) then
        return 0
    end
 
    if startYear == currentYear then
        return dayOfYear(currentYear, currentMonth, currentDay) - dayOfYear(startYear, startMonth, startDay)
    end
 
    local totalDays = 0
    
    local startDayNum = dayOfYear(startYear, startMonth, startDay)
    local daysInStartYear = isLeapYear(startYear) and 366 or 365
    totalDays = totalDays + (daysInStartYear - startDayNum)
    
    for year = startYear + 1, currentYear - 1 do
        totalDays = totalDays + (isLeapYear(year) and 366 or 365)
    end
 
    totalDays = totalDays + dayOfYear(currentYear, currentMonth, currentDay)
 
    return totalDays
end

function BriefingScreen.Show()
    if not getPlayer() or not getPlayer():isAlive() or getPlayer():isAsleep() then
        return
    end

    local playerObj = getSpecificPlayer(0); if not playerObj then return end

    local isMiseryTimeSkipActive = false
    if getCore():getGameMode() == "Misery" and MiseryTimeSkip then
        isMiseryTimeSkipActive = MiseryTimeSkip.State.isActive
    end

    if not isMiseryTimeSkipActive then
        setGameSpeed(0); UIManager.setShowPausedMessage(false)
    end

    local strings = {
        name = nil,
        date = nil,
        location = nil,
        days = nil
    }

    local startDay = getSandboxOptions():getOptionByName("StartDay"):getValue()
    local startMonth = getSandboxOptions():getOptionByName("StartMonth"):getValue()
    local startYear = getSandboxOptions():getOptionByName("StartYear"):getValue()

    local timeSinceApo = getSandboxOptions():getOptionByName("TimeSinceApo"):getValue() - 1

    local currentDay = getGameTime():getDay() + 1
    local currentMonth = getGameTime():getMonth() + 1
    local currentYear = getGameTime():getYear()

    local isDateConsistent = ((timeSinceApo == 0) and (startDay == 9 and startMonth == 7 and startYear == 1)) or ((timeSinceApo > 0) and ((currentYear - 1993) * 12 + (currentMonth - 7) >= timeSinceApo))

    local alwaysShowDays = PZAPI.ModOptions:getOptions("Briefing"):getOption("enableDaysSurvived"):getValue()
    local dateTimeFormat = PZAPI.ModOptions:getOptions("Briefing"):getOption("dateTimeFormat"):getValue()

    strings.name = playerObj:getFullName()
    
    local timeString = BriefingScreen.FormatTime(getGameTime():getTimeOfDay())
    local monthString = getText("Sandbox_StartMonth_option" .. currentMonth)
    local dateString = monthString .. " " .. currentDay .. ", " .. currentYear
    
    if dateTimeFormat == 1 then
        strings.date = timeString .. " - " .. dateString
    elseif dateTimeFormat == 2 then
        strings.date = currentYear .. ", " .. monthString .. " " .. currentDay .. " - " .. timeString
    end
    
    strings.location = BriefingScreen.GetLocationName(playerObj:getX(), playerObj:getY())

    if not alwaysShowDays and isDateConsistent then
        strings.days = string.format(getText("UI_Briefing_DaysAfterOutbreak"), BriefingScreen.GetDaysSinceOutbreak(currentDay, currentMonth, currentYear, timeSinceApo))
    else
        strings.days = string.format(getText(getGameTime():getDaysSurvived() == 1 and "UI_Briefing_Day" or "UI_Briefing_Days"), getGameTime():getDaysSurvived())
    end

    local ui = OnScreenIntroUI:new({
        strings.name,
        strings.date,
        strings.location,
        strings.days
    })

    ui:initialise()
    ui:addToUIManager()
end

function BriefingScreen.onGameStart()
    local debugAllowed = PZAPI.ModOptions:getOptions("Briefing"):getOption("debug"):getValue()

    if getCore():getGameMode() == "Tutorial" or getCore():isChallenge() or isServer() or (getDebug() and not debugAllowed and getCore():getGameMode() ~= "Misery") then
        return
    end

    if (getDebug() and not debugAllowed) and getCore():getGameMode() == "Misery" and not (MiseryTimeSkip and MiseryTimeSkip.State.isActive) then
        return
    end

    musicVolume = getCore():getOptionMusicVolume()

    delayedCallback(0, BriefingScreen.Show)
end

Events.OnGameStart.Add(BriefingScreen.onGameStart)

if PauseStart and PauseStart.OnGameStart then
    Events.OnGameStart.Remove(PauseStart.OnGameStart)
end